"use strict";
var __classPrivateFieldGet = (this && this.__classPrivateFieldGet) || function (receiver, state, kind, f) {
    if (kind === "a" && !f) throw new TypeError("Private accessor was defined without a getter");
    if (typeof state === "function" ? receiver !== state || !f : !state.has(receiver)) throw new TypeError("Cannot read private member from an object whose class did not declare it");
    return kind === "m" ? f : kind === "a" ? f.call(receiver) : f ? f.value : state.get(receiver);
};
var __classPrivateFieldSet = (this && this.__classPrivateFieldSet) || function (receiver, state, value, kind, f) {
    if (kind === "m") throw new TypeError("Private method is not writable");
    if (kind === "a" && !f) throw new TypeError("Private accessor was defined without a setter");
    if (typeof state === "function" ? receiver !== state || !f : !state.has(receiver)) throw new TypeError("Cannot write private member to an object whose class did not declare it");
    return (kind === "a" ? f.call(receiver, value) : f ? f.value = value : state.set(receiver, value)), value;
};
var _a, _ExifDateTime_fromPatterns, _ExifDateTime_looseExifFormats, _ExifDateTime_dt;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExifDateTime = void 0;
const luxon_1 = require("luxon");
const DateTime_1 = require("./DateTime");
const Maybe_1 = require("./Maybe");
const Object_1 = require("./Object");
const String_1 = require("./String");
const TimeParsing_1 = require("./TimeParsing");
const Timezones_1 = require("./Timezones");
/**
 * Encodes an ExifDateTime with an optional tz offset in minutes.
 */
class ExifDateTime {
    static from(exifOrIso, defaultZone) {
        var _b, _c;
        return exifOrIso instanceof _a
            ? exifOrIso // already an ExifDateTime
            : (0, String_1.blank)(exifOrIso)
                ? undefined // in order of strictness:
                : ((_c = (_b = this.fromExifStrict(exifOrIso, defaultZone)) !== null && _b !== void 0 ? _b : this.fromISO(exifOrIso, defaultZone)) !== null && _c !== void 0 ? _c : this.fromExifLoose(exifOrIso, defaultZone));
    }
    static fromISO(iso, defaultZone) {
        if ((0, String_1.blank)(iso) || null != iso.match(/^\d+$/))
            return undefined;
        // Unfortunately, DateTime.fromISO() is happy to parse a date with no time,
        // so we have to do this ourselves:
        return __classPrivateFieldGet(this, _a, "m", _ExifDateTime_fromPatterns).call(this, iso, (0, TimeParsing_1.timeFormats)({
            formatPrefixes: ["y-MM-dd'T'", "y-MM-dd ", "y-M-d "],
            defaultZone,
        }));
    }
    /**
     * Try to parse a date-time string from EXIF. If there is not both a date
     * and a time component, returns `undefined`.
     *
     * @param text from EXIF metadata
     * @param defaultZone a "zone name" to use as a backstop, or default, if
     * `text` doesn't specify a zone. This may be IANA-formatted, like
     * "America/Los_Angeles", or an offset, like "UTC-3". See
     * `offsetMinutesToZoneName`.
     */
    static fromEXIF(text, defaultZone) {
        var _b;
        if ((0, String_1.blank)(text))
            return undefined;
        return (
        // .fromExifStrict() uses .fromISO() as a backstop
        (_b = this.fromExifStrict(text, defaultZone)) !== null && _b !== void 0 ? _b : this.fromExifLoose(text, defaultZone));
    }
    /**
     * Parse the given date-time string, EXIF-formatted.
     *
     * @param text from EXIF metadata, in `y:M:d H:m:s` format (with optional
     * sub-seconds and/or timezone)
  
     * @param defaultZone a "zone name" to use as a backstop, or default, if
     * `text` doesn't specify a zone. This may be IANA-formatted, like
     * "America/Los_Angeles", or an offset, like "UTC-3". See
     * `offsetMinutesToZoneName`.
     */
    static fromExifStrict(text, defaultZone) {
        var _b;
        if ((0, String_1.blank)(text))
            return undefined;
        return ((_b = __classPrivateFieldGet(this, _a, "m", _ExifDateTime_fromPatterns).call(this, text, (0, TimeParsing_1.timeFormats)({ formatPrefixes: ["y:MM:dd ", "y:M:d "], defaultZone }))) !== null && _b !== void 0 ? _b : 
        // Not found yet? Maybe it's in ISO format? See
        // https://github.com/photostructure/exiftool-vendored.js/issues/71
        this.fromISO(text, defaultZone));
    }
    static fromExifLoose(text, defaultZone) {
        return (0, String_1.blank)(text)
            ? undefined
            : __classPrivateFieldGet(this, _a, "m", _ExifDateTime_fromPatterns).call(this, text, __classPrivateFieldGet(this, _a, "m", _ExifDateTime_looseExifFormats).call(this, defaultZone));
    }
    static fromDateTime(dt, opts) {
        var _b;
        if (dt == null || !dt.isValid || dt.year === 0 || dt.year === 1) {
            return undefined;
        }
        return new _a(dt.year, dt.month, dt.day, dt.hour, dt.minute, dt.second, dt.millisecond === 0 && true === (opts === null || opts === void 0 ? void 0 : opts.unsetMilliseconds)
            ? undefined
            : dt.millisecond, dt.offset === Timezones_1.UnsetZoneOffsetMinutes ? undefined : dt.offset, opts === null || opts === void 0 ? void 0 : opts.rawValue, dt.zoneName == null || ((_b = dt.zone) === null || _b === void 0 ? void 0 : _b.name) === Timezones_1.UnsetZone.name
            ? undefined
            : dt.zoneName, opts === null || opts === void 0 ? void 0 : opts.inferredZone);
    }
    /**
     * Create an ExifDateTime from a number of milliseconds since the epoch
     * (meaning since 1 January 1970 00:00:00 UTC). Uses the default zone.
     *
     * @param millis - a number of milliseconds since 1970 UTC
     *
     * @param options.rawValue - the original parsed string input
     * @param options.zone - the zone to place the DateTime into. Defaults to 'local'.
     * @param options.locale - a locale to set on the resulting DateTime instance
     * @param options.outputCalendar - the output calendar to set on the resulting DateTime instance
     * @param options.numberingSystem - the numbering system to set on the resulting DateTime instance
     */
    static fromMillis(millis, options = {}) {
        if (options.zone == null ||
            [Timezones_1.UnsetZoneName, Timezones_1.UnsetZone].includes(options.zone)) {
            delete options.zone;
        }
        let dt = luxon_1.DateTime.fromMillis(millis, {
            ...(0, Object_1.omit)(options, "rawValue"),
        });
        if (options.zone == null) {
            dt = dt.setZone(Timezones_1.UnsetZone, { keepLocalTime: true });
        }
        // TODO: is there a way to provide an invalid millisecond value?
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        return this.fromDateTime(dt, { rawValue: options.rawValue });
    }
    static now(opts = {}) {
        return this.fromMillis(Date.now(), opts);
    }
    constructor(year, month, day, hour, minute, second, millisecond, tzoffsetMinutes, rawValue, zoneName, inferredZone) {
        this.year = year;
        this.month = month;
        this.day = day;
        this.hour = hour;
        this.minute = minute;
        this.second = second;
        this.millisecond = millisecond;
        this.tzoffsetMinutes = tzoffsetMinutes;
        this.rawValue = rawValue;
        this.zoneName = zoneName;
        this.inferredZone = inferredZone;
        _ExifDateTime_dt.set(this, void 0);
        this.zone = (0, Timezones_1.getZoneName)({ zoneName, tzoffsetMinutes });
    }
    get millis() {
        return this.millisecond;
    }
    get hasZone() {
        return this.zone != null;
    }
    get unsetMilliseconds() {
        return this.millisecond == null;
    }
    setZone(zone, opts) {
        var _b;
        const dt = (0, TimeParsing_1.setZone)({
            zone,
            src: this.toDateTime(),
            srcHasZone: this.hasZone,
            opts,
        });
        return _a.fromDateTime(dt, {
            rawValue: this.rawValue,
            unsetMilliseconds: this.millisecond == null,
            inferredZone: (_b = opts === null || opts === void 0 ? void 0 : opts.inferredZone) !== null && _b !== void 0 ? _b : true,
        });
    }
    /**
     * CAUTION: This instance will inherit the system timezone if this instance
     * has an unset zone (as Luxon doesn't support "unset" timezones)
     */
    toDateTime(overrideZone) {
        var _b;
        return (__classPrivateFieldSet(this, _ExifDateTime_dt, (_b = __classPrivateFieldGet(this, _ExifDateTime_dt, "f")) !== null && _b !== void 0 ? _b : luxon_1.DateTime.fromObject({
            year: this.year,
            month: this.month,
            day: this.day,
            hour: this.hour,
            minute: this.minute,
            second: this.second,
            millisecond: this.millisecond,
        }, {
            zone: overrideZone !== null && overrideZone !== void 0 ? overrideZone : this.zone,
        }), "f"));
    }
    toEpochSeconds(overrideZone) {
        return this.toDateTime(overrideZone).toUnixInteger();
    }
    toDate() {
        return this.toDateTime().toJSDate();
    }
    toISOString(options = {}) {
        var _b;
        return (0, Maybe_1.denull)(this.toDateTime().toISO({
            suppressMilliseconds: (_b = options.suppressMilliseconds) !== null && _b !== void 0 ? _b : this.millisecond == null,
            includeOffset: this.hasZone && options.includeOffset !== false,
        }));
    }
    toExifString() {
        return (0, DateTime_1.dateTimeToExif)(this.toDateTime(), {
            includeOffset: this.hasZone,
            includeMilliseconds: this.millisecond != null,
        });
    }
    toString() {
        return this.toISOString();
    }
    /**
     * @return the epoch milliseconds of this
     */
    toMillis() {
        return this.toDateTime().toMillis();
    }
    get isValid() {
        return this.toDateTime().isValid;
    }
    toJSON() {
        return {
            _ctor: "ExifDateTime",
            year: this.year,
            month: this.month,
            day: this.day,
            hour: this.hour,
            minute: this.minute,
            second: this.second,
            millisecond: this.millisecond,
            tzoffsetMinutes: this.tzoffsetMinutes,
            rawValue: this.rawValue,
            zoneName: this.zoneName,
            inferredZone: this.inferredZone,
        };
    }
    /**
     * @return a new ExifDateTime from the given JSON. Note that this instance **may not be valid**.
     */
    static fromJSON(json) {
        return new _a(json.year, json.month, json.day, json.hour, json.minute, json.second, json.millisecond, json.tzoffsetMinutes, json.rawValue, json.zoneName, json.inferredZone);
    }
    maybeMatchZone(target, maxDeltaMs = 14 * DateTime_1.MinuteMs) {
        var _b, _c, _d;
        const targetZone = target.zone;
        if (targetZone == null || !target.hasZone)
            return;
        return ((_c = (_b = this.setZone(targetZone, { keepLocalTime: false })) === null || _b === void 0 ? void 0 : _b.ifClose(target, maxDeltaMs)) !== null && _c !== void 0 ? _c : (_d = this.setZone(targetZone, { keepLocalTime: true })) === null || _d === void 0 ? void 0 : _d.ifClose(target, maxDeltaMs));
    }
    ifClose(target, maxDeltaMs = 14 * DateTime_1.MinuteMs) {
        const ts = this.toMillis();
        const targetTs = target.toMillis();
        return Math.abs(ts - targetTs) <= maxDeltaMs ? this : undefined;
    }
    plus(duration) {
        let dt = this.toDateTime().plus(duration);
        if (!this.hasZone) {
            dt = dt.setZone(Timezones_1.UnsetZone, { keepLocalTime: true });
        }
        return _a.fromDateTime(dt, this);
    }
}
exports.ExifDateTime = ExifDateTime;
_a = ExifDateTime, _ExifDateTime_dt = new WeakMap(), _ExifDateTime_fromPatterns = function _ExifDateTime_fromPatterns(text, fmts) {
    const result = (0, TimeParsing_1.parseDateTime)(text, fmts);
    return result == null
        ? undefined
        : _a.fromDateTime(result.dt, {
            rawValue: text,
            unsetMilliseconds: result.unsetMilliseconds,
            inferredZone: result.inferredZone,
        });
}, _ExifDateTime_looseExifFormats = function* _ExifDateTime_looseExifFormats(defaultZone) {
    // The following are from actual datestamps seen in the wild (!!)
    const formats = [
        "MMM d y HH:mm:ss",
        "MMM d y, HH:mm:ss",
        // Thu Oct 13 00:12:27 2016:
        "ccc MMM d HH:mm:ss y",
    ];
    const zone = (0, String_1.notBlank)(defaultZone) ? defaultZone : Timezones_1.UnsetZone;
    for (const fmt of formats) {
        yield { fmt: fmt, zone, inferredZone: true };
    }
};
//# sourceMappingURL=ExifDateTime.js.map